/**
 * NEWSLETTER SIGNUP WIDGET
 * ========================
 * Beautiful email capture forms with multiple layouts.
 *
 * Config options:
 * - layout: 'inline' | 'stacked' | 'popup' | 'slide-in' | 'bar'
 * - title: Headline text
 * - description: Supporting text
 * - buttonText: CTA button text
 * - placeholder: Input placeholder
 * - submitUrl: API endpoint for submissions
 * - successMessage: Message after signup
 * - theme: 'light' | 'dark' | 'gradient'
 * - accentColor: Button/highlight color
 * - showName: Include name field
 * - popupTrigger: 'time' | 'scroll' | 'exit'
 * - popupDelay: ms before popup shows
 * - popupScrollPercent: Scroll % to trigger
 */

window.FaresWidgets = window.FaresWidgets || {};

window.FaresWidgets.newsletter_signup = {
    defaultConfig: {
        layout: 'stacked',
        title: 'Stay Updated',
        description: 'Get the latest tips, insights, and exclusive offers delivered to your inbox.',
        buttonText: 'Subscribe',
        placeholder: 'Enter your email',
        submitUrl: '',
        successMessage: 'Thanks for subscribing! Check your inbox.',
        theme: 'light',
        accentColor: '#2563eb',
        borderRadius: '12px',
        showName: false,
        namePlaceholder: 'Your name',
        popupTrigger: 'time',
        popupDelay: 5000,
        popupScrollPercent: 50,
        showPrivacyNote: true,
        privacyText: 'We respect your privacy. Unsubscribe anytime.',
        incentive: '',
        incentiveIcon: '🎁'
    },

    init: function(container, config) {
        this.container = container;
        this.config = { ...this.defaultConfig, ...config };
        this.render();
    },

    render: function() {
        this.addStyles();

        const { layout } = this.config;

        switch (layout) {
            case 'popup':
                this.renderPopup();
                break;
            case 'slide-in':
                this.renderSlideIn();
                break;
            case 'bar':
                this.renderBar();
                break;
            case 'inline':
                this.renderInline();
                break;
            default:
                this.renderStacked();
        }
    },

    addStyles: function() {
        if (document.getElementById('fares-newsletter-styles')) return;

        const { accentColor, borderRadius } = this.config;

        const styles = document.createElement('style');
        styles.id = 'fares-newsletter-styles';
        styles.textContent = `
            .fares-newsletter {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                --accent: ${accentColor};
                --radius: ${borderRadius};
            }
            .fares-newsletter-light {
                background: #fff;
                color: #111827;
            }
            .fares-newsletter-dark {
                background: #1f2937;
                color: #f9fafb;
            }
            .fares-newsletter-gradient {
                background: linear-gradient(135deg, ${accentColor} 0%, #7c3aed 100%);
                color: #fff;
            }
            .fares-newsletter-card {
                padding: 32px;
                border-radius: var(--radius);
                text-align: center;
            }
            .fares-newsletter-light .fares-newsletter-card {
                border: 1px solid #e5e7eb;
            }
            .fares-newsletter-dark .fares-newsletter-card {
                box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            }
            .fares-newsletter-gradient .fares-newsletter-card {
                box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            }
            .fares-newsletter-incentive {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 8px 16px;
                background: rgba(255,255,255,0.15);
                border-radius: 20px;
                font-size: 14px;
                font-weight: 500;
                margin-bottom: 16px;
            }
            .fares-newsletter-light .fares-newsletter-incentive {
                background: #fef3c7;
                color: #92400e;
            }
            .fares-newsletter-title {
                font-size: 24px;
                font-weight: 700;
                margin-bottom: 12px;
            }
            .fares-newsletter-desc {
                font-size: 15px;
                opacity: 0.9;
                margin-bottom: 24px;
                max-width: 400px;
                margin-left: auto;
                margin-right: auto;
            }
            .fares-newsletter-light .fares-newsletter-desc {
                color: #6b7280;
            }
            .fares-newsletter-form {
                display: flex;
                flex-direction: column;
                gap: 12px;
                max-width: 400px;
                margin: 0 auto;
            }
            .fares-newsletter-form.inline {
                flex-direction: row;
            }
            .fares-newsletter-input {
                flex: 1;
                padding: 14px 18px;
                font-size: 15px;
                border: 2px solid transparent;
                border-radius: 8px;
                outline: none;
                transition: all 0.2s;
            }
            .fares-newsletter-light .fares-newsletter-input {
                background: #f9fafb;
                border-color: #e5e7eb;
                color: #111827;
            }
            .fares-newsletter-light .fares-newsletter-input:focus {
                border-color: var(--accent);
            }
            .fares-newsletter-dark .fares-newsletter-input,
            .fares-newsletter-gradient .fares-newsletter-input {
                background: rgba(255,255,255,0.1);
                border-color: rgba(255,255,255,0.2);
                color: #fff;
            }
            .fares-newsletter-dark .fares-newsletter-input::placeholder,
            .fares-newsletter-gradient .fares-newsletter-input::placeholder {
                color: rgba(255,255,255,0.6);
            }
            .fares-newsletter-btn {
                padding: 14px 28px;
                font-size: 15px;
                font-weight: 600;
                border: none;
                border-radius: 8px;
                cursor: pointer;
                transition: all 0.2s;
                display: flex;
                align-items: center;
                justify-content: center;
                gap: 8px;
            }
            .fares-newsletter-light .fares-newsletter-btn {
                background: var(--accent);
                color: #fff;
            }
            .fares-newsletter-dark .fares-newsletter-btn,
            .fares-newsletter-gradient .fares-newsletter-btn {
                background: #fff;
                color: var(--accent);
            }
            .fares-newsletter-btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }
            .fares-newsletter-btn:disabled {
                opacity: 0.6;
                cursor: not-allowed;
                transform: none;
            }
            .fares-newsletter-privacy {
                font-size: 12px;
                opacity: 0.7;
                margin-top: 16px;
            }
            .fares-newsletter-success {
                display: flex;
                flex-direction: column;
                align-items: center;
                gap: 16px;
                padding: 24px;
            }
            .fares-newsletter-success-icon {
                width: 64px;
                height: 64px;
                background: #dcfce7;
                color: #166534;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 32px;
            }
            .fares-newsletter-gradient .fares-newsletter-success-icon {
                background: rgba(255,255,255,0.2);
                color: #fff;
            }
            .fares-newsletter-success-text {
                font-size: 18px;
                font-weight: 500;
            }

            /* Popup */
            .fares-newsletter-popup-overlay {
                position: fixed;
                inset: 0;
                background: rgba(0,0,0,0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 99999;
                opacity: 0;
                visibility: hidden;
                transition: all 0.3s;
            }
            .fares-newsletter-popup-overlay.visible {
                opacity: 1;
                visibility: visible;
            }
            .fares-newsletter-popup {
                max-width: 500px;
                width: 90%;
                position: relative;
                transform: scale(0.9);
                transition: transform 0.3s;
            }
            .fares-newsletter-popup-overlay.visible .fares-newsletter-popup {
                transform: scale(1);
            }
            .fares-newsletter-popup-close {
                position: absolute;
                top: 16px;
                right: 16px;
                width: 32px;
                height: 32px;
                border: none;
                background: rgba(0,0,0,0.1);
                border-radius: 50%;
                font-size: 20px;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10;
            }
            .fares-newsletter-dark .fares-newsletter-popup-close,
            .fares-newsletter-gradient .fares-newsletter-popup-close {
                background: rgba(255,255,255,0.2);
                color: #fff;
            }

            /* Bar */
            .fares-newsletter-bar {
                position: fixed;
                bottom: 0;
                left: 0;
                right: 0;
                padding: 16px 24px;
                display: flex;
                align-items: center;
                justify-content: center;
                gap: 24px;
                flex-wrap: wrap;
                z-index: 99998;
                box-shadow: 0 -4px 20px rgba(0,0,0,0.1);
            }
            .fares-newsletter-bar .fares-newsletter-form {
                flex-direction: row;
                max-width: none;
            }
            .fares-newsletter-bar .fares-newsletter-title {
                font-size: 16px;
                margin: 0;
            }
            .fares-newsletter-bar-close {
                background: none;
                border: none;
                font-size: 24px;
                cursor: pointer;
                opacity: 0.6;
            }

            /* Slide-in */
            .fares-newsletter-slidein {
                position: fixed;
                bottom: 20px;
                right: 20px;
                width: 360px;
                z-index: 99998;
                transform: translateX(120%);
                transition: transform 0.4s ease;
            }
            .fares-newsletter-slidein.visible {
                transform: translateX(0);
            }

            @media (max-width: 600px) {
                .fares-newsletter-form.inline {
                    flex-direction: column;
                }
                .fares-newsletter-bar {
                    flex-direction: column;
                    text-align: center;
                }
            }
        `;
        document.head.appendChild(styles);
    },

    renderStacked: function() {
        const { title, description, theme, showName, placeholder, namePlaceholder, buttonText, showPrivacyNote, privacyText, incentive, incentiveIcon } = this.config;

        this.container.innerHTML = `
            <div class="fares-newsletter fares-newsletter-${theme}">
                <div class="fares-newsletter-card">
                    ${incentive ? `
                        <div class="fares-newsletter-incentive">
                            <span>${incentiveIcon}</span>
                            <span>${incentive}</span>
                        </div>
                    ` : ''}
                    <h3 class="fares-newsletter-title">${title}</h3>
                    <p class="fares-newsletter-desc">${description}</p>
                    <form class="fares-newsletter-form">
                        ${showName ? `
                            <input type="text" name="name" class="fares-newsletter-input" placeholder="${namePlaceholder}" required>
                        ` : ''}
                        <input type="email" name="email" class="fares-newsletter-input" placeholder="${placeholder}" required>
                        <button type="submit" class="fares-newsletter-btn">
                            ${buttonText}
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M5 12h14M12 5l7 7-7 7"/>
                            </svg>
                        </button>
                    </form>
                    ${showPrivacyNote ? `<p class="fares-newsletter-privacy">${privacyText}</p>` : ''}
                </div>
            </div>
        `;

        this.attachFormEvents();
    },

    renderInline: function() {
        const { title, theme, placeholder, buttonText } = this.config;

        this.container.innerHTML = `
            <div class="fares-newsletter fares-newsletter-${theme}">
                <div class="fares-newsletter-card" style="padding: 24px;">
                    <div style="display: flex; align-items: center; gap: 24px; flex-wrap: wrap; justify-content: center;">
                        <h3 class="fares-newsletter-title" style="margin: 0;">${title}</h3>
                        <form class="fares-newsletter-form inline" style="flex: 1; min-width: 300px;">
                            <input type="email" name="email" class="fares-newsletter-input" placeholder="${placeholder}" required>
                            <button type="submit" class="fares-newsletter-btn">${buttonText}</button>
                        </form>
                    </div>
                </div>
            </div>
        `;

        this.attachFormEvents();
    },

    renderPopup: function() {
        const { theme, popupTrigger, popupDelay, popupScrollPercent } = this.config;

        // Create overlay
        const overlay = document.createElement('div');
        overlay.className = 'fares-newsletter-popup-overlay';
        overlay.innerHTML = `
            <div class="fares-newsletter fares-newsletter-${theme} fares-newsletter-popup">
                <button class="fares-newsletter-popup-close">×</button>
                ${this.getFormHTML()}
            </div>
        `;
        document.body.appendChild(overlay);
        this.overlay = overlay;

        // Close handlers
        overlay.querySelector('.fares-newsletter-popup-close').addEventListener('click', () => this.closePopup());
        overlay.addEventListener('click', (e) => {
            if (e.target === overlay) this.closePopup();
        });

        // Trigger logic
        if (popupTrigger === 'time') {
            setTimeout(() => this.showPopup(), popupDelay);
        } else if (popupTrigger === 'scroll') {
            window.addEventListener('scroll', () => {
                const scrolled = (window.scrollY / (document.body.scrollHeight - window.innerHeight)) * 100;
                if (scrolled >= popupScrollPercent && !this.popupShown) {
                    this.showPopup();
                }
            });
        } else if (popupTrigger === 'exit') {
            document.addEventListener('mouseleave', (e) => {
                if (e.clientY < 0 && !this.popupShown) {
                    this.showPopup();
                }
            });
        }

        this.attachFormEvents(overlay);
    },

    renderSlideIn: function() {
        const { theme, popupDelay } = this.config;

        const slideIn = document.createElement('div');
        slideIn.className = `fares-newsletter fares-newsletter-${theme} fares-newsletter-slidein`;
        slideIn.innerHTML = `
            <div class="fares-newsletter-card">
                <button class="fares-newsletter-popup-close" style="position:absolute;top:12px;right:12px;">×</button>
                ${this.getFormHTML()}
            </div>
        `;
        document.body.appendChild(slideIn);
        this.slideIn = slideIn;

        slideIn.querySelector('.fares-newsletter-popup-close').addEventListener('click', () => {
            slideIn.classList.remove('visible');
        });

        setTimeout(() => slideIn.classList.add('visible'), popupDelay);
        this.attachFormEvents(slideIn);
    },

    renderBar: function() {
        const { title, theme, placeholder, buttonText } = this.config;

        const bar = document.createElement('div');
        bar.className = `fares-newsletter fares-newsletter-${theme} fares-newsletter-bar`;
        bar.innerHTML = `
            <h3 class="fares-newsletter-title">${title}</h3>
            <form class="fares-newsletter-form">
                <input type="email" name="email" class="fares-newsletter-input" placeholder="${placeholder}" required>
                <button type="submit" class="fares-newsletter-btn">${buttonText}</button>
            </form>
            <button class="fares-newsletter-bar-close">×</button>
        `;
        document.body.appendChild(bar);

        bar.querySelector('.fares-newsletter-bar-close').addEventListener('click', () => bar.remove());
        this.attachFormEvents(bar);
    },

    getFormHTML: function() {
        const { title, description, showName, placeholder, namePlaceholder, buttonText, showPrivacyNote, privacyText, incentive, incentiveIcon } = this.config;

        return `
            <div class="fares-newsletter-card">
                ${incentive ? `
                    <div class="fares-newsletter-incentive">
                        <span>${incentiveIcon}</span>
                        <span>${incentive}</span>
                    </div>
                ` : ''}
                <h3 class="fares-newsletter-title">${title}</h3>
                <p class="fares-newsletter-desc">${description}</p>
                <form class="fares-newsletter-form">
                    ${showName ? `<input type="text" name="name" class="fares-newsletter-input" placeholder="${namePlaceholder}" required>` : ''}
                    <input type="email" name="email" class="fares-newsletter-input" placeholder="${placeholder}" required>
                    <button type="submit" class="fares-newsletter-btn">${buttonText}</button>
                </form>
                ${showPrivacyNote ? `<p class="fares-newsletter-privacy">${privacyText}</p>` : ''}
            </div>
        `;
    },

    showPopup: function() {
        if (sessionStorage.getItem('fares-newsletter-dismissed')) return;
        this.popupShown = true;
        this.overlay.classList.add('visible');
    },

    closePopup: function() {
        this.overlay.classList.remove('visible');
        sessionStorage.setItem('fares-newsletter-dismissed', 'true');
    },

    attachFormEvents: function(container = this.container) {
        const form = container.querySelector('form');
        if (!form) return;

        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            await this.handleSubmit(form, container);
        });
    },

    async handleSubmit(form, container) {
        const { submitUrl, successMessage } = this.config;
        const btn = form.querySelector('button');
        const originalText = btn.innerHTML;

        btn.disabled = true;
        btn.innerHTML = '<span class="fares-spinner"></span> Subscribing...';

        const formData = new FormData(form);
        const data = Object.fromEntries(formData.entries());

        try {
            if (submitUrl) {
                await fetch(submitUrl, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(data)
                });
            }

            // Show success
            const card = container.querySelector('.fares-newsletter-card');
            card.innerHTML = `
                <div class="fares-newsletter-success">
                    <div class="fares-newsletter-success-icon">✓</div>
                    <div class="fares-newsletter-success-text">${successMessage}</div>
                </div>
            `;
        } catch (error) {
            btn.disabled = false;
            btn.innerHTML = originalText;
            alert('Something went wrong. Please try again.');
        }
    }
};
