/**
 * FAQ ACCORDION WIDGET
 * ====================
 * Collapsible FAQ section with search and categories.
 *
 * Config options:
 * - items: Array of {question, answer, category}
 * - allowMultiple: Allow multiple items open
 * - showSearch: Enable search functionality
 * - showCategories: Show category filter
 * - defaultOpen: Index of default open item
 * - theme: 'light' | 'dark' | 'bordered' | 'minimal'
 * - iconStyle: 'plus' | 'arrow' | 'chevron'
 * - accentColor: Highlight color
 */

window.FaresWidgets = window.FaresWidgets || {};

window.FaresWidgets.faq_accordion = {
    defaultConfig: {
        items: [
            {
                question: "What services do you offer?",
                answer: "We offer comprehensive construction services including kitchen remodeling, bathroom renovation, ADU construction, roofing, flooring, and whole-home remodels. Our team handles projects of all sizes throughout San Diego County.",
                category: "Services"
            },
            {
                question: "How long does a typical project take?",
                answer: "Project timelines vary based on scope. A bathroom remodel typically takes 2-3 weeks, kitchen remodels 4-6 weeks, and ADU construction 4-6 months. We provide detailed timelines during our free consultation.",
                category: "Timeline"
            },
            {
                question: "Do you offer free estimates?",
                answer: "Yes! We offer free, no-obligation estimates for all projects. Our team will visit your home, discuss your vision, and provide a detailed quote within 48 hours.",
                category: "Pricing"
            },
            {
                question: "Are you licensed and insured?",
                answer: "Absolutely. We are fully licensed (CSLB #1054602), bonded, and insured. We also carry workers' compensation insurance for your peace of mind.",
                category: "Company"
            },
            {
                question: "What areas do you serve?",
                answer: "We serve all of San Diego County including San Diego, La Jolla, Del Mar, Carlsbad, Encinitas, Oceanside, Escondido, Poway, and surrounding communities.",
                category: "Services"
            },
            {
                question: "What payment options do you accept?",
                answer: "We accept various payment methods including credit cards, checks, and bank transfers. For larger projects, we offer financing options with approved credit.",
                category: "Pricing"
            }
        ],
        allowMultiple: false,
        showSearch: true,
        showCategories: true,
        defaultOpen: null,
        theme: 'bordered',
        iconStyle: 'chevron',
        accentColor: '#2563eb',
        borderRadius: '12px',
        animationDuration: 300
    },

    init: function(container, config) {
        this.container = container;
        this.config = { ...this.defaultConfig, ...config };
        this.openItems = new Set();
        if (this.config.defaultOpen !== null) {
            this.openItems.add(this.config.defaultOpen);
        }
        this.render();
        this.attachEvents();
    },

    render: function() {
        this.addStyles();

        const { items, showSearch, showCategories, theme } = this.config;
        const categories = [...new Set(items.map(i => i.category).filter(Boolean))];

        this.container.innerHTML = `
            <div class="fares-faq fares-faq-${theme}">
                ${showSearch ? `
                    <div class="fares-faq-search">
                        <svg class="fares-faq-search-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="11" cy="11" r="8"/>
                            <path d="M21 21l-4.35-4.35"/>
                        </svg>
                        <input type="text" class="fares-faq-search-input" placeholder="Search questions...">
                    </div>
                ` : ''}

                ${showCategories && categories.length > 1 ? `
                    <div class="fares-faq-categories">
                        <button class="fares-faq-category active" data-category="all">All</button>
                        ${categories.map(cat => `
                            <button class="fares-faq-category" data-category="${cat}">${cat}</button>
                        `).join('')}
                    </div>
                ` : ''}

                <div class="fares-faq-items">
                    ${items.map((item, index) => this.renderItem(item, index)).join('')}
                </div>
            </div>
        `;
    },

    renderItem: function(item, index) {
        const { iconStyle } = this.config;
        const isOpen = this.openItems.has(index);

        const icons = {
            plus: { closed: '+', open: '−' },
            arrow: { closed: '→', open: '↓' },
            chevron: { closed: '›', open: '⌄' }
        };

        return `
            <div class="fares-faq-item ${isOpen ? 'open' : ''}" data-index="${index}" data-category="${item.category || ''}">
                <button class="fares-faq-question">
                    <span class="fares-faq-question-text">${item.question}</span>
                    <span class="fares-faq-icon">${isOpen ? icons[iconStyle].open : icons[iconStyle].closed}</span>
                </button>
                <div class="fares-faq-answer" style="${isOpen ? '' : 'display: none;'}">
                    <div class="fares-faq-answer-content">${item.answer}</div>
                </div>
            </div>
        `;
    },

    addStyles: function() {
        if (document.getElementById('fares-faq-styles')) return;

        const { accentColor, borderRadius, animationDuration } = this.config;

        const styles = document.createElement('style');
        styles.id = 'fares-faq-styles';
        styles.textContent = `
            .fares-faq {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                --accent: ${accentColor};
                --radius: ${borderRadius};
                --duration: ${animationDuration}ms;
            }

            .fares-faq-search {
                position: relative;
                margin-bottom: 24px;
            }
            .fares-faq-search-icon {
                position: absolute;
                left: 16px;
                top: 50%;
                transform: translateY(-50%);
                width: 20px;
                height: 20px;
                color: #9ca3af;
            }
            .fares-faq-search-input {
                width: 100%;
                padding: 14px 16px 14px 48px;
                font-size: 15px;
                border: 1px solid #e5e7eb;
                border-radius: var(--radius);
                outline: none;
                transition: border-color 0.2s, box-shadow 0.2s;
            }
            .fares-faq-search-input:focus {
                border-color: var(--accent);
                box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
            }

            .fares-faq-categories {
                display: flex;
                flex-wrap: wrap;
                gap: 8px;
                margin-bottom: 24px;
            }
            .fares-faq-category {
                padding: 8px 16px;
                font-size: 14px;
                border: 1px solid #e5e7eb;
                background: #fff;
                border-radius: 20px;
                cursor: pointer;
                transition: all 0.2s;
            }
            .fares-faq-category:hover {
                border-color: var(--accent);
                color: var(--accent);
            }
            .fares-faq-category.active {
                background: var(--accent);
                border-color: var(--accent);
                color: #fff;
            }

            .fares-faq-items {
                display: flex;
                flex-direction: column;
                gap: 12px;
            }

            /* Bordered theme */
            .fares-faq-bordered .fares-faq-item {
                border: 1px solid #e5e7eb;
                border-radius: var(--radius);
                overflow: hidden;
            }
            .fares-faq-bordered .fares-faq-item.open {
                border-color: var(--accent);
            }

            /* Light theme */
            .fares-faq-light .fares-faq-item {
                background: #f9fafb;
                border-radius: var(--radius);
            }

            /* Dark theme */
            .fares-faq-dark {
                background: #1f2937;
                padding: 24px;
                border-radius: var(--radius);
            }
            .fares-faq-dark .fares-faq-item {
                background: #374151;
                border-radius: var(--radius);
            }
            .fares-faq-dark .fares-faq-question-text,
            .fares-faq-dark .fares-faq-answer-content {
                color: #f9fafb;
            }
            .fares-faq-dark .fares-faq-search-input {
                background: #374151;
                border-color: #4b5563;
                color: #f9fafb;
            }
            .fares-faq-dark .fares-faq-category {
                background: #374151;
                border-color: #4b5563;
                color: #f9fafb;
            }

            /* Minimal theme */
            .fares-faq-minimal .fares-faq-item {
                border-bottom: 1px solid #e5e7eb;
                border-radius: 0;
            }
            .fares-faq-minimal .fares-faq-item:last-child {
                border-bottom: none;
            }

            .fares-faq-question {
                width: 100%;
                display: flex;
                justify-content: space-between;
                align-items: center;
                padding: 18px 20px;
                font-size: 16px;
                font-weight: 500;
                text-align: left;
                background: none;
                border: none;
                cursor: pointer;
                transition: background 0.2s;
            }
            .fares-faq-question:hover {
                background: rgba(0,0,0,0.02);
            }
            .fares-faq-question-text {
                color: #111827;
                flex: 1;
                padding-right: 16px;
            }
            .fares-faq-icon {
                font-size: 20px;
                color: var(--accent);
                transition: transform var(--duration);
            }
            .fares-faq-item.open .fares-faq-icon {
                transform: rotate(180deg);
            }

            .fares-faq-answer {
                overflow: hidden;
                transition: max-height var(--duration) ease;
            }
            .fares-faq-answer-content {
                padding: 0 20px 20px;
                font-size: 15px;
                line-height: 1.7;
                color: #4b5563;
            }

            .fares-faq-item.hidden {
                display: none;
            }
        `;
        document.head.appendChild(styles);
    },

    attachEvents: function() {
        // Toggle items
        this.container.querySelectorAll('.fares-faq-question').forEach(btn => {
            btn.addEventListener('click', () => {
                const item = btn.closest('.fares-faq-item');
                const index = parseInt(item.dataset.index);
                this.toggle(index);
            });
        });

        // Search
        const searchInput = this.container.querySelector('.fares-faq-search-input');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => this.search(e.target.value));
        }

        // Categories
        this.container.querySelectorAll('.fares-faq-category').forEach(btn => {
            btn.addEventListener('click', () => {
                this.container.querySelectorAll('.fares-faq-category').forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                this.filterByCategory(btn.dataset.category);
            });
        });
    },

    toggle: function(index) {
        const { allowMultiple } = this.config;
        const item = this.container.querySelector(`[data-index="${index}"]`);
        const answer = item.querySelector('.fares-faq-answer');
        const icon = item.querySelector('.fares-faq-icon');

        if (this.openItems.has(index)) {
            // Close
            this.openItems.delete(index);
            item.classList.remove('open');
            answer.style.display = 'none';
            icon.textContent = this.getIcon(false);
        } else {
            // Close others if not allowMultiple
            if (!allowMultiple) {
                this.openItems.forEach(i => {
                    const otherItem = this.container.querySelector(`[data-index="${i}"]`);
                    otherItem.classList.remove('open');
                    otherItem.querySelector('.fares-faq-answer').style.display = 'none';
                    otherItem.querySelector('.fares-faq-icon').textContent = this.getIcon(false);
                });
                this.openItems.clear();
            }

            // Open
            this.openItems.add(index);
            item.classList.add('open');
            answer.style.display = 'block';
            icon.textContent = this.getIcon(true);
        }
    },

    getIcon: function(isOpen) {
        const icons = {
            plus: { closed: '+', open: '−' },
            arrow: { closed: '→', open: '↓' },
            chevron: { closed: '›', open: '⌄' }
        };
        return isOpen ? icons[this.config.iconStyle].open : icons[this.config.iconStyle].closed;
    },

    search: function(query) {
        const items = this.container.querySelectorAll('.fares-faq-item');
        query = query.toLowerCase();

        items.forEach(item => {
            const question = item.querySelector('.fares-faq-question-text').textContent.toLowerCase();
            const answer = item.querySelector('.fares-faq-answer-content').textContent.toLowerCase();

            if (question.includes(query) || answer.includes(query)) {
                item.classList.remove('hidden');
            } else {
                item.classList.add('hidden');
            }
        });
    },

    filterByCategory: function(category) {
        const items = this.container.querySelectorAll('.fares-faq-item');

        items.forEach(item => {
            if (category === 'all' || item.dataset.category === category) {
                item.classList.remove('hidden');
            } else {
                item.classList.add('hidden');
            }
        });
    }
};
