/**
 * COUNTDOWN TIMER WIDGET
 * ======================
 * Urgency-creating countdown timer for sales and events.
 * Research shows countdown timers increase conversions by 14.4%.
 *
 * Config options:
 * - endDate: Target date/time (ISO string or timestamp)
 * - type: 'fixed' | 'evergreen' | 'daily'
 * - evergreenMinutes: For evergreen, minutes to count down
 * - dailyResetTime: For daily, time to reset (HH:MM)
 * - title: Text above countdown
 * - expiredText: Text to show when expired
 * - expiredAction: 'hide' | 'show_text' | 'redirect'
 * - redirectUrl: URL to redirect to when expired
 * - style: 'boxes' | 'inline' | 'minimal' | 'flip'
 * - showDays: true/false
 * - showHours: true/false
 * - showMinutes: true/false
 * - showSeconds: true/false
 * - primaryColor: Main color
 * - backgroundColor: Background color
 * - textColor: Text color
 * - labelColor: Label text color
 * - size: 'small' | 'medium' | 'large'
 * - position: 'inline' | 'sticky-top' | 'sticky-bottom'
 */

window.FaresWidgets = window.FaresWidgets || {};

window.FaresWidgets.countdown_timer = {
    defaultConfig: {
        endDate: null,
        type: 'fixed',
        evergreenMinutes: 30,
        dailyResetTime: '00:00',
        title: 'Offer ends in:',
        expiredText: 'This offer has expired',
        expiredAction: 'show_text',
        redirectUrl: '',
        style: 'boxes',
        showDays: true,
        showHours: true,
        showMinutes: true,
        showSeconds: true,
        primaryColor: '#e53935',
        backgroundColor: '#ffffff',
        textColor: '#333333',
        labelColor: '#666666',
        size: 'medium',
        position: 'inline',
        borderRadius: '8px',
        showUrgency: true
    },

    init: function(container, config) {
        this.container = container;
        this.config = { ...this.defaultConfig, ...config };
        this.calculateEndDate();
        this.render();
        this.startCountdown();
    },

    calculateEndDate: function() {
        const { type, endDate, evergreenMinutes, dailyResetTime } = this.config;

        switch (type) {
            case 'evergreen':
                // Check for stored end time
                const stored = localStorage.getItem('fares-countdown-' + this.config.title);
                if (stored && new Date(stored) > new Date()) {
                    this.endTime = new Date(stored);
                } else {
                    this.endTime = new Date(Date.now() + evergreenMinutes * 60 * 1000);
                    localStorage.setItem('fares-countdown-' + this.config.title, this.endTime.toISOString());
                }
                break;

            case 'daily':
                const [hours, minutes] = dailyResetTime.split(':').map(Number);
                this.endTime = new Date();
                this.endTime.setHours(hours, minutes, 0, 0);
                if (this.endTime <= new Date()) {
                    this.endTime.setDate(this.endTime.getDate() + 1);
                }
                break;

            default: // fixed
                this.endTime = endDate ? new Date(endDate) : new Date(Date.now() + 24 * 60 * 60 * 1000);
        }
    },

    render: function() {
        const {
            title, style, showDays, showHours, showMinutes, showSeconds,
            primaryColor, backgroundColor, textColor, labelColor,
            size, position, borderRadius
        } = this.config;

        this.addStyles();

        // Size classes
        const sizeMap = { small: 'sm', medium: 'md', large: 'lg' };
        const sizeClass = sizeMap[size] || 'md';

        // Position styles
        let positionStyle = '';
        if (position === 'sticky-top') {
            positionStyle = 'position:fixed;top:0;left:0;right:0;z-index:9999;';
        } else if (position === 'sticky-bottom') {
            positionStyle = 'position:fixed;bottom:0;left:0;right:0;z-index:9999;';
        }

        this.container.innerHTML = `
            <div class="fares-countdown fares-countdown-${style} fares-countdown-${sizeClass}"
                 style="background:${backgroundColor};color:${textColor};border-radius:${borderRadius};${positionStyle}">

                ${title ? `<div class="fares-countdown-title">${title}</div>` : ''}

                <div class="fares-countdown-timer">
                    ${showDays ? `
                        <div class="fares-countdown-unit" data-unit="days">
                            <span class="fares-countdown-value" style="background:${primaryColor}">00</span>
                            <span class="fares-countdown-label" style="color:${labelColor}">Days</span>
                        </div>
                    ` : ''}
                    ${showHours ? `
                        <div class="fares-countdown-unit" data-unit="hours">
                            <span class="fares-countdown-value" style="background:${primaryColor}">00</span>
                            <span class="fares-countdown-label" style="color:${labelColor}">Hours</span>
                        </div>
                    ` : ''}
                    ${showMinutes ? `
                        <div class="fares-countdown-unit" data-unit="minutes">
                            <span class="fares-countdown-value" style="background:${primaryColor}">00</span>
                            <span class="fares-countdown-label" style="color:${labelColor}">Mins</span>
                        </div>
                    ` : ''}
                    ${showSeconds ? `
                        <div class="fares-countdown-unit" data-unit="seconds">
                            <span class="fares-countdown-value" style="background:${primaryColor}">00</span>
                            <span class="fares-countdown-label" style="color:${labelColor}">Secs</span>
                        </div>
                    ` : ''}
                </div>

                <div class="fares-countdown-expired" style="display:none;">
                    ${this.config.expiredText}
                </div>
            </div>
        `;

        // Store references
        this.timerEl = this.container.querySelector('.fares-countdown-timer');
        this.expiredEl = this.container.querySelector('.fares-countdown-expired');
        this.units = {
            days: this.container.querySelector('[data-unit="days"] .fares-countdown-value'),
            hours: this.container.querySelector('[data-unit="hours"] .fares-countdown-value'),
            minutes: this.container.querySelector('[data-unit="minutes"] .fares-countdown-value'),
            seconds: this.container.querySelector('[data-unit="seconds"] .fares-countdown-value')
        };
    },

    addStyles: function() {
        if (document.getElementById('fares-countdown-styles')) return;

        const styles = document.createElement('style');
        styles.id = 'fares-countdown-styles';
        styles.textContent = `
            .fares-countdown {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                padding: 20px;
                text-align: center;
            }
            .fares-countdown-title {
                font-size: 16px;
                font-weight: 500;
                margin-bottom: 12px;
            }
            .fares-countdown-timer {
                display: flex;
                justify-content: center;
                gap: 12px;
            }
            .fares-countdown-unit {
                display: flex;
                flex-direction: column;
                align-items: center;
            }
            .fares-countdown-boxes .fares-countdown-value {
                min-width: 60px;
                padding: 12px 8px;
                font-size: 28px;
                font-weight: 700;
                color: #fff;
                border-radius: 8px;
                display: block;
            }
            .fares-countdown-boxes .fares-countdown-label {
                font-size: 12px;
                text-transform: uppercase;
                margin-top: 6px;
            }
            .fares-countdown-inline .fares-countdown-timer {
                gap: 4px;
            }
            .fares-countdown-inline .fares-countdown-value {
                font-size: 24px;
                font-weight: 700;
                padding: 0 4px;
                background: none !important;
                color: inherit;
            }
            .fares-countdown-inline .fares-countdown-label {
                display: none;
            }
            .fares-countdown-inline .fares-countdown-unit::after {
                content: ':';
                font-size: 24px;
                font-weight: 700;
                margin-left: 4px;
            }
            .fares-countdown-inline .fares-countdown-unit:last-child::after {
                display: none;
            }
            .fares-countdown-minimal .fares-countdown-value {
                font-size: 36px;
                font-weight: 300;
                padding: 0;
                background: none !important;
                color: inherit;
            }
            .fares-countdown-minimal .fares-countdown-label {
                font-size: 11px;
                text-transform: uppercase;
            }
            .fares-countdown-sm .fares-countdown-value { font-size: 20px !important; min-width: 44px !important; }
            .fares-countdown-sm .fares-countdown-label { font-size: 10px !important; }
            .fares-countdown-lg .fares-countdown-value { font-size: 40px !important; min-width: 80px !important; }
            .fares-countdown-lg .fares-countdown-label { font-size: 14px !important; }
            .fares-countdown-expired {
                font-size: 18px;
                font-weight: 500;
                padding: 20px;
            }
            .fares-countdown-urgent .fares-countdown-value {
                animation: fares-countdown-pulse 1s ease-in-out infinite;
            }
            @keyframes fares-countdown-pulse {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(1.05); }
            }
        `;
        document.head.appendChild(styles);
    },

    startCountdown: function() {
        this.update();
        this.interval = setInterval(() => this.update(), 1000);
    },

    update: function() {
        const now = new Date();
        const diff = this.endTime - now;

        if (diff <= 0) {
            this.handleExpired();
            return;
        }

        // Calculate time units
        const days = Math.floor(diff / (1000 * 60 * 60 * 24));
        const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((diff % (1000 * 60)) / 1000);

        // Update display
        if (this.units.days) this.units.days.textContent = String(days).padStart(2, '0');
        if (this.units.hours) this.units.hours.textContent = String(hours).padStart(2, '0');
        if (this.units.minutes) this.units.minutes.textContent = String(minutes).padStart(2, '0');
        if (this.units.seconds) this.units.seconds.textContent = String(seconds).padStart(2, '0');

        // Add urgency effect when < 1 hour
        if (this.config.showUrgency && diff < 60 * 60 * 1000) {
            this.container.querySelector('.fares-countdown').classList.add('fares-countdown-urgent');
        }
    },

    handleExpired: function() {
        clearInterval(this.interval);

        const { expiredAction, redirectUrl } = this.config;

        switch (expiredAction) {
            case 'hide':
                this.container.style.display = 'none';
                break;
            case 'redirect':
                if (redirectUrl) window.location.href = redirectUrl;
                break;
            default: // show_text
                this.timerEl.style.display = 'none';
                this.expiredEl.style.display = 'block';
        }
    }
};
