/**
 * COOKIE CONSENT WIDGET
 * =====================
 * GDPR/CCPA compliant cookie consent banner.
 *
 * Config options:
 * - position: 'bottom' | 'top' | 'bottom-left' | 'bottom-right'
 * - layout: 'bar' | 'box' | 'modal'
 * - theme: 'light' | 'dark' | 'auto'
 * - showPreferences: Allow category preferences
 * - categories: Cookie categories with descriptions
 * - policyUrl: Privacy policy link
 * - accentColor: Button color
 */

window.FaresWidgets = window.FaresWidgets || {};

window.FaresWidgets.cookie_consent = {
    defaultConfig: {
        position: 'bottom',
        layout: 'bar',
        theme: 'light',
        title: 'We value your privacy',
        description: 'We use cookies to enhance your browsing experience, serve personalized content, and analyze our traffic. By clicking "Accept All", you consent to our use of cookies.',
        showPreferences: true,
        categories: [
            {
                id: 'necessary',
                name: 'Necessary',
                description: 'Essential for the website to function properly. Cannot be disabled.',
                required: true,
                default: true
            },
            {
                id: 'analytics',
                name: 'Analytics',
                description: 'Help us understand how visitors interact with our website.',
                required: false,
                default: true
            },
            {
                id: 'marketing',
                name: 'Marketing',
                description: 'Used to deliver personalized advertisements.',
                required: false,
                default: false
            },
            {
                id: 'preferences',
                name: 'Preferences',
                description: 'Remember your settings and preferences.',
                required: false,
                default: true
            }
        ],
        policyUrl: '/privacy-policy',
        acceptAllText: 'Accept All',
        acceptSelectedText: 'Accept Selected',
        rejectAllText: 'Reject All',
        settingsText: 'Cookie Settings',
        saveText: 'Save Preferences',
        accentColor: '#2563eb',
        borderRadius: '12px',
        cookieName: 'cookie_consent',
        cookieExpiry: 365, // days
        showFloatingButton: true,
        onAccept: null, // callback function
        onReject: null // callback function
    },

    init: function(container, config) {
        this.config = { ...this.defaultConfig, ...config };
        this.preferences = this.loadPreferences();

        if (!this.preferences) {
            this.render();
        } else if (this.config.showFloatingButton) {
            this.renderFloatingButton();
        }
    },

    loadPreferences: function() {
        const cookie = document.cookie.split(';').find(c => c.trim().startsWith(this.config.cookieName + '='));
        if (cookie) {
            try {
                return JSON.parse(decodeURIComponent(cookie.split('=')[1]));
            } catch (e) {
                return null;
            }
        }
        return null;
    },

    savePreferences: function(preferences) {
        const expires = new Date(Date.now() + this.config.cookieExpiry * 24 * 60 * 60 * 1000).toUTCString();
        document.cookie = `${this.config.cookieName}=${encodeURIComponent(JSON.stringify(preferences))}; expires=${expires}; path=/; SameSite=Lax`;
        this.preferences = preferences;
    },

    render: function() {
        this.addStyles();

        const { position, layout, theme, title, description, showPreferences, policyUrl, acceptAllText, rejectAllText, settingsText } = this.config;

        const banner = document.createElement('div');
        banner.className = `fares-cc fares-cc-${position} fares-cc-${layout}`;
        banner.dataset.theme = theme;
        banner.innerHTML = `
            <div class="fares-cc-container">
                <div class="fares-cc-content">
                    <div class="fares-cc-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-1 14H9v-2h2v2zm0-4H9V7h2v5zm4 4h-2v-2h2v2zm0-4h-2V7h2v5z"/>
                        </svg>
                    </div>
                    <div class="fares-cc-text">
                        <h3 class="fares-cc-title">${title}</h3>
                        <p class="fares-cc-desc">${description}</p>
                        <a href="${policyUrl}" class="fares-cc-link" target="_blank">Learn more</a>
                    </div>
                </div>
                <div class="fares-cc-actions">
                    ${showPreferences ? `<button class="fares-cc-btn fares-cc-btn-secondary" data-action="settings">${settingsText}</button>` : ''}
                    <button class="fares-cc-btn fares-cc-btn-secondary" data-action="reject">${rejectAllText}</button>
                    <button class="fares-cc-btn fares-cc-btn-primary" data-action="accept">${acceptAllText}</button>
                </div>
            </div>
        `;

        document.body.appendChild(banner);
        this.banner = banner;
        this.attachEvents();

        // Animation
        requestAnimationFrame(() => {
            banner.classList.add('visible');
        });
    },

    addStyles: function() {
        if (document.getElementById('fares-cc-styles')) return;

        const { accentColor, borderRadius } = this.config;

        const styles = document.createElement('style');
        styles.id = 'fares-cc-styles';
        styles.textContent = `
            .fares-cc {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                --accent: ${accentColor};
                --radius: ${borderRadius};
                position: fixed;
                z-index: 99999;
                padding: 20px;
                transform: translateY(100%);
                transition: transform 0.4s ease;
            }
            .fares-cc.visible {
                transform: translateY(0);
            }
            .fares-cc-bottom { bottom: 0; left: 0; right: 0; }
            .fares-cc-top { top: 0; left: 0; right: 0; transform: translateY(-100%); }
            .fares-cc-top.visible { transform: translateY(0); }
            .fares-cc-bottom-left { bottom: 0; left: 0; max-width: 420px; }
            .fares-cc-bottom-right { bottom: 0; right: 0; max-width: 420px; }

            .fares-cc[data-theme="light"] {
                --bg: #fff;
                --text: #111827;
                --text-muted: #6b7280;
                --border: #e5e7eb;
            }
            .fares-cc[data-theme="dark"] {
                --bg: #1f2937;
                --text: #f9fafb;
                --text-muted: #9ca3af;
                --border: #374151;
            }

            .fares-cc-container {
                background: var(--bg);
                border-radius: var(--radius);
                box-shadow: 0 -4px 20px rgba(0,0,0,0.1);
                padding: 20px 24px;
            }

            .fares-cc-bar .fares-cc-container {
                display: flex;
                align-items: center;
                gap: 24px;
                flex-wrap: wrap;
            }
            .fares-cc-bar .fares-cc-content {
                flex: 1;
                display: flex;
                align-items: center;
                gap: 16px;
            }

            .fares-cc-box .fares-cc-container {
                max-width: 420px;
            }
            .fares-cc-box .fares-cc-actions {
                margin-top: 20px;
            }

            .fares-cc-icon {
                flex-shrink: 0;
                width: 40px;
                height: 40px;
                background: rgba(37, 99, 235, 0.1);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            .fares-cc-icon svg {
                width: 22px;
                height: 22px;
                color: var(--accent);
            }

            .fares-cc-title {
                font-size: 16px;
                font-weight: 600;
                color: var(--text);
                margin-bottom: 4px;
            }
            .fares-cc-desc {
                font-size: 14px;
                color: var(--text-muted);
                line-height: 1.5;
                margin: 0;
            }
            .fares-cc-link {
                font-size: 13px;
                color: var(--accent);
                text-decoration: none;
            }
            .fares-cc-link:hover {
                text-decoration: underline;
            }

            .fares-cc-actions {
                display: flex;
                gap: 10px;
                flex-wrap: wrap;
            }

            .fares-cc-btn {
                padding: 10px 20px;
                border-radius: 8px;
                font-size: 14px;
                font-weight: 500;
                cursor: pointer;
                transition: all 0.2s;
            }
            .fares-cc-btn-primary {
                background: var(--accent);
                color: #fff;
                border: none;
            }
            .fares-cc-btn-primary:hover {
                opacity: 0.9;
            }
            .fares-cc-btn-secondary {
                background: transparent;
                color: var(--text);
                border: 1px solid var(--border);
            }
            .fares-cc-btn-secondary:hover {
                background: var(--border);
            }

            /* Settings Modal */
            .fares-cc-modal-overlay {
                position: fixed;
                inset: 0;
                background: rgba(0,0,0,0.5);
                z-index: 100000;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 20px;
            }
            .fares-cc-modal {
                background: var(--bg);
                border-radius: var(--radius);
                max-width: 500px;
                width: 100%;
                max-height: 80vh;
                overflow-y: auto;
            }
            .fares-cc-modal-header {
                padding: 20px 24px;
                border-bottom: 1px solid var(--border);
                display: flex;
                align-items: center;
                justify-content: space-between;
            }
            .fares-cc-modal-title {
                font-size: 18px;
                font-weight: 600;
                color: var(--text);
            }
            .fares-cc-modal-close {
                width: 32px;
                height: 32px;
                border: none;
                background: none;
                cursor: pointer;
                font-size: 24px;
                color: var(--text-muted);
            }
            .fares-cc-modal-body {
                padding: 20px 24px;
            }
            .fares-cc-category {
                padding: 16px 0;
                border-bottom: 1px solid var(--border);
            }
            .fares-cc-category:last-child {
                border-bottom: none;
            }
            .fares-cc-category-header {
                display: flex;
                align-items: center;
                justify-content: space-between;
            }
            .fares-cc-category-name {
                font-weight: 600;
                color: var(--text);
                font-size: 15px;
            }
            .fares-cc-category-desc {
                font-size: 13px;
                color: var(--text-muted);
                margin-top: 6px;
            }
            .fares-cc-toggle {
                position: relative;
                width: 44px;
                height: 24px;
                background: #e5e7eb;
                border-radius: 12px;
                cursor: pointer;
                transition: background 0.3s;
            }
            .fares-cc-toggle.on {
                background: var(--accent);
            }
            .fares-cc-toggle.disabled {
                opacity: 0.5;
                cursor: not-allowed;
            }
            .fares-cc-toggle::after {
                content: '';
                position: absolute;
                width: 20px;
                height: 20px;
                background: #fff;
                border-radius: 50%;
                top: 2px;
                left: 2px;
                transition: transform 0.3s;
                box-shadow: 0 1px 3px rgba(0,0,0,0.2);
            }
            .fares-cc-toggle.on::after {
                transform: translateX(20px);
            }
            .fares-cc-modal-footer {
                padding: 16px 24px;
                border-top: 1px solid var(--border);
                display: flex;
                gap: 12px;
                justify-content: flex-end;
            }

            /* Floating Button */
            .fares-cc-floating {
                position: fixed;
                bottom: 20px;
                left: 20px;
                width: 44px;
                height: 44px;
                background: var(--bg);
                border: 1px solid var(--border);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                cursor: pointer;
                box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                z-index: 99998;
                transition: transform 0.2s;
            }
            .fares-cc-floating:hover {
                transform: scale(1.1);
            }
            .fares-cc-floating svg {
                width: 20px;
                height: 20px;
                color: var(--text-muted);
            }

            @media (max-width: 768px) {
                .fares-cc-bar .fares-cc-content {
                    flex-direction: column;
                    text-align: center;
                }
                .fares-cc-bar .fares-cc-actions {
                    width: 100%;
                    justify-content: center;
                }
            }
        `;
        document.head.appendChild(styles);
    },

    attachEvents: function() {
        // Accept all
        this.banner.querySelector('[data-action="accept"]').addEventListener('click', () => {
            const preferences = {};
            this.config.categories.forEach(cat => {
                preferences[cat.id] = true;
            });
            this.savePreferences(preferences);
            this.close();
            if (this.config.onAccept) this.config.onAccept(preferences);
        });

        // Reject all
        this.banner.querySelector('[data-action="reject"]').addEventListener('click', () => {
            const preferences = {};
            this.config.categories.forEach(cat => {
                preferences[cat.id] = cat.required;
            });
            this.savePreferences(preferences);
            this.close();
            if (this.config.onReject) this.config.onReject(preferences);
        });

        // Settings
        const settingsBtn = this.banner.querySelector('[data-action="settings"]');
        if (settingsBtn) {
            settingsBtn.addEventListener('click', () => this.showSettings());
        }
    },

    showSettings: function() {
        const { theme, categories, acceptSelectedText, saveText } = this.config;

        // Initialize preferences for settings
        const currentPrefs = {};
        categories.forEach(cat => {
            currentPrefs[cat.id] = cat.required || cat.default;
        });

        const modal = document.createElement('div');
        modal.className = 'fares-cc-modal-overlay';
        modal.dataset.theme = theme;
        modal.innerHTML = `
            <div class="fares-cc-modal">
                <div class="fares-cc-modal-header">
                    <h3 class="fares-cc-modal-title">Cookie Preferences</h3>
                    <button class="fares-cc-modal-close">&times;</button>
                </div>
                <div class="fares-cc-modal-body">
                    ${categories.map(cat => `
                        <div class="fares-cc-category">
                            <div class="fares-cc-category-header">
                                <span class="fares-cc-category-name">${cat.name}</span>
                                <div class="fares-cc-toggle ${cat.required || cat.default ? 'on' : ''} ${cat.required ? 'disabled' : ''}" data-category="${cat.id}"></div>
                            </div>
                            <p class="fares-cc-category-desc">${cat.description}</p>
                        </div>
                    `).join('')}
                </div>
                <div class="fares-cc-modal-footer">
                    <button class="fares-cc-btn fares-cc-btn-secondary" data-action="cancel">Cancel</button>
                    <button class="fares-cc-btn fares-cc-btn-primary" data-action="save">${saveText}</button>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        this.modal = modal;

        // Toggle events
        modal.querySelectorAll('.fares-cc-toggle:not(.disabled)').forEach(toggle => {
            toggle.addEventListener('click', () => {
                toggle.classList.toggle('on');
                currentPrefs[toggle.dataset.category] = toggle.classList.contains('on');
            });
        });

        // Cancel
        modal.querySelector('[data-action="cancel"]').addEventListener('click', () => {
            modal.remove();
        });

        // Close button
        modal.querySelector('.fares-cc-modal-close').addEventListener('click', () => {
            modal.remove();
        });

        // Overlay click
        modal.addEventListener('click', (e) => {
            if (e.target === modal) modal.remove();
        });

        // Save
        modal.querySelector('[data-action="save"]').addEventListener('click', () => {
            modal.querySelectorAll('.fares-cc-toggle').forEach(toggle => {
                currentPrefs[toggle.dataset.category] = toggle.classList.contains('on');
            });
            this.savePreferences(currentPrefs);
            modal.remove();
            this.close();
            if (this.config.onAccept) this.config.onAccept(currentPrefs);
        });
    },

    close: function() {
        this.banner.classList.remove('visible');
        setTimeout(() => {
            this.banner.remove();
            if (this.config.showFloatingButton) {
                this.renderFloatingButton();
            }
        }, 400);
    },

    renderFloatingButton: function() {
        const { theme } = this.config;

        const btn = document.createElement('button');
        btn.className = 'fares-cc-floating';
        btn.dataset.theme = theme;
        btn.innerHTML = `
            <svg viewBox="0 0 24 24" fill="currentColor">
                <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-1 14H9v-2h2v2zm0-4H9V7h2v5zm4 4h-2v-2h2v2zm0-4h-2V7h2v5z"/>
            </svg>
        `;
        btn.title = 'Cookie Settings';
        btn.addEventListener('click', () => {
            btn.remove();
            this.render();
        });

        document.body.appendChild(btn);
    },

    // Public API
    getPreferences: function() {
        return this.preferences;
    },

    hasConsent: function(category) {
        return this.preferences && this.preferences[category] === true;
    }
};
